package raycast;

/*******************************************************************************
* <p>Title: Project Raycast</p>
*
* @author Bernhard Schulz (www.bsyte.de)
* @version 1.0
*******************************************************************************/
public class Arc
{
  private static Arc _instance = null;

  private int factor = 0;
  private int arc360 = 0;
  private int arc270 = 0;
  private int arc180 = 0;
  private int arc90  = 0;
  private int arc0   = 0;

  private double[] m_sin = null;
  private double[] m_cos = null;
  private double[] m_tan = null;

/*******************************************************************************
*
*******************************************************************************/
public static void initInstance( )
  {
  if (_instance == null)
    _instance = new Arc();
  }

/*******************************************************************************
* Initialisiert die Singleton-Instanz
* @param f Faktor * 360 Grad, Damit werden die trigonom. Tabellen generiert
*******************************************************************************/
public static void initInstance(int f)
  {
  if (f < 1)
    f=1;

  if (_instance == null)
    _instance = new Arc(f);
  }

/*******************************************************************************
* Gibt die referenz auf das Singleton-Objekt zurueck
* @return referenz auf das Objekt
*******************************************************************************/
public static Arc getInstance()
  {
  if (_instance == null)
    _instance = new Arc();
  return (_instance);
  }

/*******************************************************************************
* @brief conctructor
*******************************************************************************/
private Arc()
  {
  factor = 1;
  init();
  }

/*******************************************************************************
* Konstruktor
* @param f Faktor fuer die Tabellengenerierung
*******************************************************************************/
private Arc(int f)
  {
  factor = f;
  init();
  }

/*******************************************************************************
*
*******************************************************************************/
private void init()
  {
  arc360 = 360 * factor;
  arc270 = 270 * factor;
  arc180 = 180 * factor;
  arc90  =  90 * factor;
  arc0   =   0;

  m_sin = new double[arc360 + 1];
  m_cos = new double[arc360 + 1];
  m_tan = new double[arc360 + 1];

  int i;
  for (i=0; i<=arc360; i++)
     {
     m_sin[i] = Math.sin(grad2rad(i));
     m_cos[i] = Math.cos(grad2rad(i));
     m_tan[i] = Math.tan(grad2rad(i));
    }
  }

/*******************************************************************************
* Gibt den sin-Wert zurueck
* @param w Winkel Grad (0 - 360) * f !!!
* @return sin-Wert [Grad]
*******************************************************************************/
public double sin(int w)  {  return (m_sin[w]);  }

/*******************************************************************************
* Gibt den cos-Wert zurueck
* @param w Winkel Grad (0 - 360) * f !!!
* @return cos-Wert [Grad]
*******************************************************************************/
public double cos(int w)  {  return (m_cos[w]);  }

/*******************************************************************************
* Gibt den tan zurueck
* @param w Winkel Grad (0 - 360) * f !!!
* @return tan-Wert [Grad]
*******************************************************************************/
public double tan(int w)  {  return (m_tan[w]);  }

/*******************************************************************************
* Pythagoras
* @param  x Ankathete
* @param  y Gegenkathete
* @return l Laenge der Hypothenuse
*******************************************************************************/
public double len(double x, double y)  { return (Math.sqrt((x*x) + (y*y)));  }

/*******************************************************************************
* Normiert einen Winkel zwischen '0' und '360' Grad
* @param w Eingangswinkel
* @return normierter Winkel
*******************************************************************************/
public int norm(int w)
  {
  int erg = w;

  if (w >= arc360)
      erg = w - arc360;
  else if (w < 0)
      erg = arc360 + w;

  return (erg);
  }

/*******************************************************************************
* Gibt den Quadranten zurueck
* @return Wert fuer '0' Grad
*******************************************************************************/
public int getQuadrant(int w)
{
  if ((w >= 0)   && (w < 90))  return(1); // 1. Quadrant
  if ((w >= 90)  && (w < 180)) return(2); // 2. Quadrant
  if ((w >= 180) && (w < 270)) return(3); // 3. Quadrant
  if ((w >= 270) && (w < 360)) return(4); // 4. Quadrant
  return (1);
}

/*******************************************************************************
* Gibt den Winkel zurueck
* @return
*******************************************************************************/
public int getWinkel(int x0, int y0, int x1, int y1)
{
  int dx = x1 - x0;
  int dy = y1 - y0;

  double w = 0;
  int ww = 0;

  if (dx == 0)
     {
     if (dy > 0)
        ww = 90;
     else
        ww = 270;
     }
  else
     {
     double q = (double)dy / (double)dx;
     double qq =  Math.atan(q);
     w = (qq * 180.0) / Math.PI;
     ww = (int)w;

     if ((dx < 0) && (dy >= 0))
        ww = ww + 180;
     else if ((dx < 0) && (dy < 0))
        ww = ww + 180;
     else if ((dx > 0) && (dy < 0))
        ww = ww + 360;
     }

  return (ww);
}

/*******************************************************************************
* Gibt den Wert fuer '0' Grad zurueck
* @return Wert fuer '0' Grad
*******************************************************************************/
public int getARC0()  { return (arc0); }

/*******************************************************************************
* Gibt den Wert fuer '90' Grad zurueck
* @return Wert fuer '90' Grad
*******************************************************************************/
public int getARC90()  { return (arc90); }

/*******************************************************************************
* Gibt den Wert fuer '180' Grad zurueck
* @return Wert fuer '180' Grad
*******************************************************************************/
public int getARC180()  { return (arc180); }

/*******************************************************************************
* Gibt den Wert fuer '270' Grad zurueck
* @return Wert fuer '270' Grad
*******************************************************************************/
public int getARC270()  { return (arc270); }

/*******************************************************************************
* Gibt den Wert fuer '360' Grad zurueck
* @return Wert fuer '360' Grad
*******************************************************************************/
public int getARC360()  { return (arc360); }

/*******************************************************************************
* Skaliert einen Winkel (0 bis 360) in die durch f bestimmte Aufloesung
* @param w Winkel (0 bis 360)
* @return Skalierter Winkel
*******************************************************************************/
public  int scale(int w) { return (w*factor); }

/*******************************************************************************
* Hilfsfunktion: Umwandlung rad->grad
* @param grad Winkel in [rad]
* @return Winkel in [grad]
*******************************************************************************/
private double grad2rad(double grad) { return ((Math.PI * grad) / arc180); }
}
